<?php
require '../includes/db.php';
include '../includes/header.php';

// Seguridad
if (!isset($_SESSION['user_id'])) { header("Location: ../index.php"); exit; }

$msg = '';
$error = '';

// --- PROCESAR FORMULARIO (Crear / Editar / Eliminar) ---
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $action = $_POST['action'];

    try {
        $pdo->beginTransaction(); // Usamos transacciones para asegurar el stock

        if ($action == 'create') {
            // --- CREAR CONSUMO ---
            $barbero_id = $_POST['barbero_id'];
            $producto_id = $_POST['producto_id'];
            $cantidad = $_POST['cantidad'];

            // Verificar Stock
            $stmt = $pdo->prepare("SELECT precio_venta, stock, nombre FROM products WHERE id = ? FOR UPDATE");
            $stmt->execute([$producto_id]);
            $prod = $stmt->fetch();

            if ($prod && $prod['stock'] >= $cantidad) {
                $monto_total = $prod['precio_venta'] * $cantidad;
                
                // Insertar
                $stmtIns = $pdo->prepare("INSERT INTO barber_consumptions (barbero_id, producto_id, cantidad, monto_total) VALUES (?, ?, ?, ?)");
                $stmtIns->execute([$barbero_id, $producto_id, $cantidad, $monto_total]);

                // Descontar Stock
                $stmtUpd = $pdo->prepare("UPDATE products SET stock = stock - ? WHERE id = ?");
                $stmtUpd->execute([$cantidad, $producto_id]);
                
                $msg = "Consumo registrado correctamente.";
            } else {
                $error = "Stock insuficiente para " . $prod['nombre'];
            }

        } elseif ($action == 'delete') {
            // --- ELIMINAR CONSUMO (Devolver Stock) ---
            $id = $_POST['id'];

            // Obtener datos del consumo antes de borrar para saber qué devolver
            $stmtGet = $pdo->prepare("SELECT producto_id, cantidad FROM barber_consumptions WHERE id = ?");
            $stmtGet->execute([$id]);
            $consumo = $stmtGet->fetch();

            if ($consumo) {
                // Devolver stock al inventario
                $stmtRestock = $pdo->prepare("UPDATE products SET stock = stock + ? WHERE id = ?");
                $stmtRestock->execute([$consumo['cantidad'], $consumo['producto_id']]);

                // Borrar registro
                $stmtDel = $pdo->prepare("DELETE FROM barber_consumptions WHERE id = ?");
                $stmtDel->execute([$id]);
                
                $msg = "Consumo eliminado y stock restaurado.";
            }

        } elseif ($action == 'edit') {
            // --- EDITAR CONSUMO (Complejo: Devolver viejo -> Restar nuevo) ---
            $id = $_POST['id'];
            $nuevo_barbero_id = $_POST['barbero_id'];
            $nuevo_producto_id = $_POST['producto_id'];
            $nueva_cantidad = $_POST['cantidad'];

            // 1. Obtener datos viejos
            $stmtGet = $pdo->prepare("SELECT producto_id, cantidad FROM barber_consumptions WHERE id = ?");
            $stmtGet->execute([$id]);
            $viejo = $stmtGet->fetch();

            if ($viejo) {
                // 2. Devolver stock viejo TEMPORALMENTE
                $stmtRestock = $pdo->prepare("UPDATE products SET stock = stock + ? WHERE id = ?");
                $stmtRestock->execute([$viejo['cantidad'], $viejo['producto_id']]);

                // 3. Verificar si hay stock para la NUEVA solicitud
                $stmtCheck = $pdo->prepare("SELECT precio_venta, stock, nombre FROM products WHERE id = ?");
                $stmtCheck->execute([$nuevo_producto_id]);
                $prodNuevo = $stmtCheck->fetch();

                if ($prodNuevo && $prodNuevo['stock'] >= $nueva_cantidad) {
                    // 4. Proceder con el cambio: Descontar nuevo stock
                    $stmtDeduct = $pdo->prepare("UPDATE products SET stock = stock - ? WHERE id = ?");
                    $stmtDeduct->execute([$nueva_cantidad, $nuevo_producto_id]);

                    // 5. Actualizar el registro de consumo
                    $nuevo_monto = $prodNuevo['precio_venta'] * $nueva_cantidad;
                    $stmtUpdate = $pdo->prepare("UPDATE barber_consumptions SET barbero_id=?, producto_id=?, cantidad=?, monto_total=? WHERE id=?");
                    $stmtUpdate->execute([$nuevo_barbero_id, $nuevo_producto_id, $nueva_cantidad, $nuevo_monto, $id]);

                    $msg = "Consumo actualizado correctamente.";
                } else {
                    // ROLLBACK MANUAL: Si no hay stock para lo nuevo, volvemos a quitar el stock viejo que devolvimos
                    $stmtRevert = $pdo->prepare("UPDATE products SET stock = stock - ? WHERE id = ?");
                    $stmtRevert->execute([$viejo['cantidad'], $viejo['producto_id']]);
                    $error = "No se pudo editar: Stock insuficiente del nuevo producto seleccionado.";
                }
            }
        }

        $pdo->commit(); // Confirmar cambios si todo salió bien

    } catch (Exception $e) {
        $pdo->rollBack();
        $error = "Error del sistema: " . $e->getMessage();
    }
}

// --- CARGAR DATOS PARA LA VISTA ---
$barberos = $pdo->query("SELECT id, nombre, apellido FROM users WHERE rol='barbero' AND activo=1")->fetchAll();
$productos = $pdo->query("SELECT * FROM products WHERE tipo IN ('snack', 'bebida') AND activo=1")->fetchAll();

// Historial completo (paginado o ultimos 50)
$historial = $pdo->query("SELECT c.*, u.nombre as b_nombre, u.apellido as b_apellido, p.nombre as p_nombre 
                          FROM barber_consumptions c 
                          JOIN users u ON c.barbero_id = u.id 
                          JOIN products p ON c.producto_id = p.id 
                          ORDER BY c.fecha_consumo DESC LIMIT 50")->fetchAll();
?>

<div class="row">
    <div class="col-md-4">
        <div class="card mb-3 border-gold">
            <div class="card-header bg-gold fw-bold text-dark">Registrar Nuevo Consumo</div>
            <div class="card-body">
                <?php if($msg) echo "<div class='alert alert-success py-2'>$msg</div>"; ?>
                <?php if($error) echo "<div class='alert alert-danger py-2'>$error</div>"; ?>
                
                <form method="POST">
                    <input type="hidden" name="action" value="create">
                    <div class="mb-3">
                        <label class="text-gold">Barbero</label>
                        <select name="barbero_id" class="form-select bg-dark text-white border-secondary" required>
                            <option value="">-- Seleccionar --</option>
                            <?php foreach($barberos as $b): ?>
                                <option value="<?php echo $b['id']; ?>"><?php echo $b['nombre'].' '.$b['apellido']; ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="text-gold">Producto</label>
                        <select name="producto_id" class="form-select bg-dark text-white border-secondary" required>
                            <option value="">-- Seleccionar --</option>
                            <?php foreach($productos as $p): ?>
                                <option value="<?php echo $p['id']; ?>">
                                    <?php echo $p['nombre']; ?> - $<?php echo $p['precio_venta']; ?> (Stock: <?php echo $p['stock']; ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="text-gold">Cantidad</label>
                        <input type="number" name="cantidad" value="1" min="1" class="form-control bg-dark text-white border-secondary" required>
                    </div>
                    <button type="submit" class="btn btn-gold w-100">Registrar y Descontar</button>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-md-8">
        <div class="card">
            <div class="card-header bg-secondary text-white">Historial de Consumos (Últimos 50)</div>
            <div class="card-body table-responsive">
                <table class="table table-dark table-hover align-middle">
                    <thead>
                        <tr>
                            <th class="text-gold">Barbero</th>
                            <th class="text-gold">Producto</th>
                            <th class="text-center">Cant.</th>
                            <th class="text-gold">Total</th>
                            <th>Fecha</th>
                            <th class="text-end">Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach($historial as $h): ?>
                        <tr>
                            <td><?php echo $h['b_nombre'] . ' ' . $h['b_apellido']; ?></td>
                            <td><?php echo $h['p_nombre']; ?></td>
                            <td class="text-center"><?php echo $h['cantidad']; ?></td>
                            <td class="text-danger fw-bold">-$<?php echo number_format($h['monto_total'], 2); ?></td>
                            <td><?php echo date('d/m H:i', strtotime($h['fecha_consumo'])); ?></td>
                            <td class="text-end">
                                <button class="btn btn-sm btn-outline-warning" 
                                        onclick="editar(<?php echo htmlspecialchars(json_encode($h)); ?>)">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <form method="POST" style="display:inline-block;" onsubmit="return confirm('¿Eliminar este consumo? El stock del producto será devuelto.');">
                                    <input type="hidden" name="action" value="delete">
                                    <input type="hidden" name="id" value="<?php echo $h['id']; ?>">
                                    <button type="submit" class="btn btn-sm btn-outline-danger"><i class="fas fa-trash"></i></button>
                                </form>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="modalEditar" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content bg-dark text-white border-warning">
            <form method="POST">
                <div class="modal-header border-secondary">
                    <h5 class="modal-title text-gold">Editar Consumo</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <input type="hidden" name="action" value="edit">
                    <input type="hidden" name="id" id="edit_id">
                    
                    <div class="alert alert-info py-1" style="font-size: 0.9em;">
                        <i class="fas fa-info-circle"></i> Al editar, el stock se ajustará automáticamente.
                    </div>

                    <div class="mb-3">
                        <label>Barbero</label>
                        <select name="barbero_id" id="edit_barbero_id" class="form-select bg-secondary text-white border-0" required>
                            <?php foreach($barberos as $b): ?>
                                <option value="<?php echo $b['id']; ?>"><?php echo $b['nombre'].' '.$b['apellido']; ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label>Producto</label>
                        <select name="producto_id" id="edit_producto_id" class="form-select bg-secondary text-white border-0" required>
                            <?php foreach($productos as $p): ?>
                                <option value="<?php echo $p['id']; ?>">
                                    <?php echo $p['nombre']; ?> ($<?php echo $p['precio_venta']; ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label>Cantidad</label>
                        <input type="number" name="cantidad" id="edit_cantidad" min="1" class="form-control" required>
                    </div>
                </div>
                <div class="modal-footer border-secondary">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-gold">Guardar Cambios</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function editar(data) {
    var modal = new bootstrap.Modal(document.getElementById('modalEditar'));
    document.getElementById('edit_id').value = data.id;
    document.getElementById('edit_barbero_id').value = data.barbero_id;
    document.getElementById('edit_producto_id').value = data.producto_id;
    document.getElementById('edit_cantidad').value = data.cantidad;
    modal.show();
}
</script>

<?php include '../includes/footer.php'; ?>