<?php
require '../includes/db.php';
header('Content-Type: application/json');

// Solo Admin y Barberos pueden vender
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['status' => 'error', 'message' => 'No autorizado']);
    exit;
}

// Recibimos los datos JSON desde el Javascript de caja.php
$input = json_decode(file_get_contents('php://input'), true);

if (!$input) {
    echo json_encode(['status' => 'error', 'message' => 'Datos inválidos']);
    exit;
}

$cliente_id = $input['cliente_id'];
$barbero_id = $input['barbero_id'];
$metodo_pago = $input['metodo_pago'];
$referencia = $input['referencia'] ?? null;
$banco = $input['banco'] ?? null;
$fecha_pago = $input['fecha_pago'] ?? date('Y-m-d');
$observaciones = $input['observaciones'] ?? '';
$items = $input['items']; // Array de productos/servicios

// 1. NUEVO: Recibir y validar la propina
$propina = isset($input['propina']) ? floatval($input['propina']) : 0;
if($propina < 0) $propina = 0;

if (empty($items) || empty($cliente_id) || empty($barbero_id)) {
    echo json_encode(['status' => 'error', 'message' => 'Faltan datos obligatorios (Cliente, Barbero o Ítems).']);
    exit;
}

try {
    $pdo->beginTransaction();

    $total_venta_items = 0; // Suma solo de productos y servicios
    
    // 2. MODIFICADO: Insertamos la propina en la cabecera
    // El 'total' lo dejamos en 0 momentáneamente, lo actualizaremos al final con la suma de todo.
    $stmt = $pdo->prepare("INSERT INTO sales (cliente_id, barbero_id, total, propina, metodo_pago, referencia_bancaria, banco_origen, fecha_pago, observaciones) VALUES (?, ?, 0, ?, ?, ?, ?, ?, ?)");
    $stmt->execute([$cliente_id, $barbero_id, $propina, $metodo_pago, $referencia, $banco, $fecha_pago, $observaciones]);
    $sale_id = $pdo->lastInsertId();

    foreach ($items as $item) {
        $tipo = $item['tipo']; // 'servicio' o 'producto'
        $id_item = $item['id'];
        $cantidad = $item['cantidad'];
        
        $comision_monto = 0;
        $precio_unitario = 0;

        if ($tipo == 'servicio') {
            // Obtener precio real y reglas de lavado
            $stmtS = $pdo->prepare("SELECT precio, requiere_lavado FROM services WHERE id = ?");
            $stmtS->execute([$id_item]);
            $servicio = $stmtS->fetch();
            
            $precio_unitario = $servicio['precio'];
            
            // LÓGICA DE NEGOCIO: 70/30 y Descuento de Lavado
            $base_calculo = $precio_unitario;
            if ($servicio['requiere_lavado'] == 1) {
                $base_calculo = $precio_unitario - 1; // Restamos $1 de insumos
                if($base_calculo < 0) $base_calculo = 0;
            }
            
            // 70% para el barbero sobre la base
            $comision_monto = $base_calculo * 0.70;

        } else {
            // Es Producto
            $stmtP = $pdo->prepare("SELECT precio_venta, stock, nombre FROM products WHERE id = ?");
            $stmtP->execute([$id_item]);
            $producto = $stmtP->fetch();
            
            if ($producto['stock'] < $cantidad) {
                throw new Exception("Stock insuficiente para: " . $producto['nombre']);
            }

            $precio_unitario = $producto['precio_venta'];
            $comision_monto = 0; // Productos no dan comisión al barbero
            
            // DESCONTAR STOCK
            $stmtUpdate = $pdo->prepare("UPDATE products SET stock = stock - ? WHERE id = ?");
            $stmtUpdate->execute([$cantidad, $id_item]);
        }

        $subtotal_linea = $precio_unitario * $cantidad;
        $total_venta_items += $subtotal_linea;
        
        // Comision total por esta linea
        $comision_total_linea = $comision_monto * $cantidad;

        // Insertar detalle
        $stmtDet = $pdo->prepare("INSERT INTO sale_details (sale_id, item_type, item_id, cantidad, precio_unitario, comision_monto) VALUES (?, ?, ?, ?, ?, ?)");
        $stmtDet->execute([$sale_id, $tipo, $id_item, $cantidad, $precio_unitario, $comision_total_linea]);
    }

    // 3. MODIFICADO: Calcular el Gran Total (Items + Propina)
    $gran_total = $total_venta_items + $propina;

    // Actualizar el total final en la cabecera
    $stmtUpdateSale = $pdo->prepare("UPDATE sales SET total = ? WHERE id = ?");
    $stmtUpdateSale->execute([$gran_total, $sale_id]);

    $pdo->commit();
    echo json_encode(['status' => 'success', 'message' => 'Venta registrada correctamente. ID: ' . $sale_id]);

} catch (Exception $e) {
    $pdo->rollBack();
    echo json_encode(['status' => 'error', 'message' => 'Error en venta: ' . $e->getMessage()]);
}
?>