-- Creación de la base de datos (opcional si ya la creaste manual)
CREATE DATABASE IF NOT EXISTS barberia_sys CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE barberia_sys;

-- 1. TABLA DE USUARIOS (Administradores y Barberos)
CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    cedula VARCHAR(20) UNIQUE NOT NULL,
    nombre VARCHAR(100) NOT NULL,
    apellido VARCHAR(100) NOT NULL,
    fecha_nacimiento DATE,
    username VARCHAR(50) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL, -- Aquí se guardará la contraseña encriptada (Hash)
    rol ENUM('admin', 'barbero') NOT NULL DEFAULT 'barbero',
    activo TINYINT(1) DEFAULT 1, -- 1 = Activo, 0 = Inactivo (para no borrar historial)
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- 2. TABLA DE CLIENTES
CREATE TABLE clients (
    id INT AUTO_INCREMENT PRIMARY KEY,
    cedula VARCHAR(20) UNIQUE NOT NULL,
    nombre VARCHAR(100) NOT NULL,
    apellido VARCHAR(100) NOT NULL,
    fecha_nacimiento DATE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- 3. TABLA DE SERVICIOS
CREATE TABLE services (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nombre VARCHAR(100) NOT NULL,
    descripcion TEXT,
    precio DECIMAL(10, 2) NOT NULL,
    requiere_lavado TINYINT(1) DEFAULT 0, -- 1 = Sí (Descontará $1 al calcular comisión), 0 = No
    activo TINYINT(1) DEFAULT 1
);

-- 4. TABLA DE PRODUCTOS (Inventario: Capilar, Snacks, Bebidas)
CREATE TABLE products (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nombre VARCHAR(100) NOT NULL,
    tipo ENUM('producto_capilar', 'snack', 'bebida') NOT NULL,
    stock INT NOT NULL DEFAULT 0,
    precio_compra DECIMAL(10, 2) NOT NULL,
    precio_venta DECIMAL(10, 2) NOT NULL,
    activo TINYINT(1) DEFAULT 1
);

-- 5. TABLA DE VENTAS (Cabecera - El Ticket General)
CREATE TABLE sales (
    id INT AUTO_INCREMENT PRIMARY KEY,
    cliente_id INT,
    barbero_id INT,
    total DECIMAL(10, 2) NOT NULL,
    metodo_pago ENUM('pago_movil', 'transferencia', 'efectivo_bs', 'efectivo_usd', 'binance', 'zelle') NOT NULL,
    referencia_bancaria VARCHAR(100) NULL, -- Nulo si es efectivo
    banco_origen VARCHAR(100) NULL,
    fecha_pago DATE, -- Fecha introducida manualmente para la referencia
    observaciones TEXT,
    fecha_creacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    estado ENUM('pagada', 'anulada') DEFAULT 'pagada',
    FOREIGN KEY (cliente_id) REFERENCES clients(id) ON DELETE SET NULL,
    FOREIGN KEY (barbero_id) REFERENCES users(id) ON DELETE SET NULL
);

-- 6. DETALLE DE VENTA (Aquí ocurre la magia de la comisión)
CREATE TABLE sale_details (
    id INT AUTO_INCREMENT PRIMARY KEY,
    sale_id INT NOT NULL,
    item_type ENUM('servicio', 'producto') NOT NULL,
    item_id INT NOT NULL, -- ID del servicio o del producto
    cantidad INT NOT NULL DEFAULT 1,
    precio_unitario DECIMAL(10, 2) NOT NULL, -- Precio al momento de la venta
    comision_monto DECIMAL(10, 2) DEFAULT 0.00, -- Aquí se guarda EXACTAMENTE cuánto ganó el barbero por este ítem
    FOREIGN KEY (sale_id) REFERENCES sales(id) ON DELETE CASCADE
);

-- 7. CONSUMO INTERNO DE BARBEROS (Bebidas/Snacks que consumen ellos)
CREATE TABLE barber_consumptions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    barbero_id INT NOT NULL,
    producto_id INT NOT NULL,
    cantidad INT NOT NULL,
    monto_total DECIMAL(10, 2) NOT NULL, -- Precio venta o costo (según tu política, usualmente precio venta)
    fecha_consumo TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    pagado TINYINT(1) DEFAULT 0, -- 0 = Pendiente de descontar en nómina, 1 = Ya descontado
    FOREIGN KEY (barbero_id) REFERENCES users(id),
    FOREIGN KEY (producto_id) REFERENCES products(id)
);

-- 8. PAGOS A BARBEROS (Historial de Nómina)
CREATE TABLE payroll_history (
    id INT AUTO_INCREMENT PRIMARY KEY,
    barbero_id INT NOT NULL,
    fecha_inicio DATE NOT NULL,
    fecha_fin DATE NOT NULL,
    total_servicios DECIMAL(10, 2), -- Total generado en servicios
    total_comisiones DECIMAL(10, 2), -- El 70% calculado
    total_descuentos_consumo DECIMAL(10, 2), -- Resta por snacks/bebidas
    monto_pagado_final DECIMAL(10, 2),
    fecha_pago TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (barbero_id) REFERENCES users(id)
);

-- USUARIO ADMINISTRADOR POR DEFECTO (Para poder entrar la primera vez)
-- Usuario: admin / Clave: admin123 (DEBES CAMBIARLA AL ENTRAR AL SISTEMA)
-- Nota: En un sistema real la clave 'admin123' debe estar encriptada con Bcrypt. 
-- Aquí la pongo plana solo como ejemplo inicial, tu código PHP debe encriptarla.
INSERT INTO users (cedula, nombre, apellido, username, password, rol, activo) 
VALUES ('00000000', 'Administrador', 'Principal', 'admin', 'admin123', 'admin', 1);